
; Revision fixes bug that stops feedback speed control after the saft start.
; TRIAC based motor controller automatic calibration for frequency (50 or 60Hz) requires approximately a 220VAC-250VAC supply
; has soft start for when the speed pot is left past zero at power up and also when pot brought to 0, soft start ends when pot rotation matches speed
; Current feedback included with a gain setting
; compensation included for phase lag in zero voltage crossing detector

;** Option to remove/restore soft start. With VR1 and VR2 rotated fully clockwise and with 5V from the VR1 CON2 terminal connected to AN0 at pin 7, apply power.
;  Each time will alternate between softstart on and off.
; default is soft start on at initial programming. 

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=12F675
	#include P12F675.inc

;Program Configuration Register 
	__CONFIG    _CPD_OFF & _CP_OFF & _BODEN_ON & _MCLRE_ON  & _PWRTE_ON & _WDT_OFF & _INTRC_OSC_NOCLKOUT	

; EEPROM
EEPROM0			equ	H'00'	; non-volatile storage for mode

; RAM 
STORE1			equ	H'20'	; delay counter	;
STORE2			equ	H'21'	; delay counter
TEMP			equ	H'22'	; temporary working
SPEED0		equ	H'23'	; speed setting by VR1 ms byte
SPEED1		equ	H'24'	; speed setting by VR1 ls byte
FEEDBACK		equ	H'25'	; current feedback value
GAIN			equ	H'26'	; feedback gain
FEED_SPED0	equ	H'27'	; feedback speed ms
FEED_SPED1	equ	H'28'	; feedback speed ls
SPEEDPOT		equ	H'29'	; speedpot read flag
GATE_OFF		equ	H'2A'	; gate off flag
GATE_ON0		equ	H'2B'	; gate switch on point
GATE_ON1		equ	H'2C'	; gate switch on point
PULSE			equ	H'2D'	; gate pulses flag
SOFTSTART		equ	H'2E'	; soft start flag
SOFT_RUN		equ	H'2F'	; soft start run up value
DEC_RATE		equ	H'30'	; decrease rate
INTER_STRT		equ	H'31'	; interrupt start counter
GATE			equ	H'32'	; first gate drive signal flag
CYCLE_9		equ	H'33'	; count waveform cycle NINETY degrees
NINETY			equ	H'34'	; 90 degree position
MULT0			equ	H'35'	; multiplier ms byts
MULT1			equ	H'36'	; multiplier ls byte
SOFT_EXTN		equ	H'37'	; soft start extension
SOFT			equ	H'38'	; soft start with no current flag

; interrupt
W_TMP			equ	H'50'	; temporary store for w in interrupt
STATUS_TMP	equ	H'51'	; temporary store of status in interrupt 

; math routines
TEMPB1		equ H'52'
TEMPB0      		equ H'53'
TEMPD       		equ H'54'   	 ; temporary storage
AARGB3		equ	H'55'
AARGB2      		equ H'56'
AARGB1      		equ H'57'
AARGB0      		equ H'58'	; most significant byte of argument A
BARGB1      		equ H'59'
BARGB0      		equ H'5A'	; most significant byte of argument B
REMB1			equ H'5B'
REMB0       		equ H'5C'   	; most significant byte of remainder
LOOPCOUNT	equ H'5D'   	; loop counter
STO1	    		equ	H'5E'	; AARGB1 storage
STO2			equ	H'5F'	; AARGB2 storage

; preprogram EEPROM DATA 
	
	ORG     H'2100'

	DE		H'0' 				;  soft start with current at 0

; ******************************************************************

; start at memory 0

	org		0			; reset vector
	goto	MAIN
; interrupt vector
	org		4
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0

	btfsc	CYCLE_9,1		; count waveform cycle for NINETY, if clear get timer0 count from edge to edge
	goto	RUN

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALIBRATION; at power up 

; calibrate values for multiplier and ninety degrees value
	btfss	INTCON,GPIF		; GP2 port change 
	goto	RECLAIM
	movf	GPIO,w				; read to clear mismatch plus check GP2 Level
	bcf		INTCON,GPIF
	btfsc 	GPIO,2
	goto 	RECLAIM
; If CYCLE_PR2 ,0 is clear then clear timer1
	btfss	CYCLE_9,0		; count waveform cycle
	goto	WAIT_NEXT_EDGE
; get timer1 count
	bcf		T1CON,0		; stop timer 1

;  timer1 count from edge to edge 
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /2
; store in multiplier
	movf	TMR1H,w		; most significant store in MULT0
	movwf	MULT0
	movf	TMR1L,w		; least significant
	movwf	MULT1
; take away 64 due to phase lag in zero voltage crossing (400us) and 100us gate pulse (8us x 64us = 512us)
	movlw	D'64'
	subwf	MULT1,f
	btfss	STATUS,C		; if carry then decrease ms byte
	decf	MULT0,f
; get 90 degree value by dividing TIMER1  by a further 16
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /2
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /4
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /8
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,w		; /16
	movwf	NINETY			; place 90 degree value
	movlw	D'7'				; compensate phase lag of zero voltage detection (64us per value)
	subwf	NINETY,f
	bsf		T1CON,0		; start timer1 again
; enable interrupts
	bsf		STATUS,RP0	; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 overflow interrupt
	bcf		STATUS,RP0	; select memory bank 0
	bsf		INTCON,PEIE	; enable periperal interrupts 
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag
	bsf		CYCLE_9,1		; enable normal run
	goto	RECLAIM
WAIT_NEXT_EDGE
	clrf 		TMR1L			; cleared on edge
	clrf		TMR1H
	bsf		CYCLE_9,0		; set bit 0
	goto	RECLAIM
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; check which interrupt	
RUN	
	btfsc	PIR1,TMR1IF		; timer 1 interrupt flag
	goto	T1	
	btfss	INTCON,GPIF	; GP2 port change interrupt flag 
	goto	RECLAIM

EDGE
;  (GP2)	
	movf	GPIO,w			; clear mismatch
	bcf		INTCON,GPIF	; flag cleared
	clrf		TMR0			; set at 0	
	clrw
	movwf	GPIO			; gate off at bit 5 (GP5)
	clrf		PULSE
	clrf		GATE
; preload timer with 65536 minus the speed setting with feedback

	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	movf	GATE_ON0,w	; Value to load into TMR1H
	movwf	TMR1H			 ; Write High byte
	movf	GATE_ON1,w	 ; Value to load into TMR1L
	movwf	TMR1L	 		;  Write Low byte
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	clrf		PULSE			; flag for gate pulses
	goto	RECLAIM

T1 ; timer1 overflow	
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag

	btfss	SPEEDPOT,0	; speed pot read flag. If  flag is set as having been read then continue, otherwise exit
	goto	RECLAIM

; check if first flag clear
	btfsc	GATE,0			; trigger on first gate 
	goto	CHK_PULSES
; first pulse. Use wider gate pulse after 90 degrees
	
	movf	TMR0,w
	subwf	NINETY,w		; 
	btfss	STATUS,C		; wider gate pulse if timer greater
	goto	WIDER	

CHK_PULSES
	btfsc	PULSE,0
	goto	PULSE_OFF

; drive gate at timer overflow
; multiple gate pulses for trigger at between 0-90 degrees
; single gate pulse for when trigger is between 90 and 180 degrees
; timer0 has 64us per count
; Lag due to filter on zero crossing is 8 counts 
; value in ninety compensates for zero voltage crossing lag of around 500us. Timer0 counts at 64us about 8 counts
	movf	TMR0,w
	subwf	NINETY,w		;~ D'70'; for 50Hz	D'57'; for 60Hz
	btfss	STATUS,C		; bypass gate on if timer greater
	goto	INC_PULSE	
; check if startup period has ended
	movf	INTER_STRT	,w	; interrupt start counter
	btfsc	STATUS,Z		; if zero then can drive gate
	goto	DRIVE
	decfsz	INTER_STRT,f	; decrease until zero
	goto	INC_PULSE		; bypass drive
DRIVE
	movlw	B'00000000'		; gate off if flag clear
	btfsc	GATE_OFF,0	; gate off flag. If set allow gate drive
	movlw	B'00100000'		; set bit 5
	movwf	GPIO			; gate drive 
INC_PULSE
	incf		PULSE,f			; once first fired alternate on and off
	bsf		GATE,0			; set after first gate pulse

; reload for 40us
	movlw	H'FE'			; H' FE'  for 40us (use H'F7' for 100us or H'DF' for 300us wider single pulse test)**
	movwf	TMR1L
	movlw	H'FF'
	movwf	TMR1H
	goto	RECLAIM

WIDER
	movlw	B'00000000'		; gate off if flag clear
	btfsc	GATE_OFF,0	; gate off flag. If set allow gate drive
	movlw	B'00100000'		; set bit 5
	movwf	GPIO			; gate drive
	incf		PULSE,f			; once first fired alternate on and off
	bsf		GATE,0			; set after first gate pulse

; reload for 100us gate on
	movlw	H'F7'
	movwf	TMR1L
	movlw	H'FF'
	movwf	TMR1H
	goto	RECLAIM

PULSE_OFF
	clrf		PULSE			; remark out to prevent multiple pulses. ie a single 40us pulse < 90 degrees, 100us pulse >90 degrees. Change 40us pulse see above)**
	clrw
	movwf	GPIO			; gate off
; reload for 200us gate off
	movlw	H'EB'
	movwf	TMR1L
	movlw	H'FF'
	movwf	TMR1H
	
; end of interrupt reclaim w and status 

RECLAIM
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf  	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt

; ***********************************************************
		
MAIN
; set oscillator calibration
	bsf		STATUS,RP0	; bank 1
	call		H'3FF'			; oscillator calibration value
	movwf	OSCCAL
	bcf		STATUS,RP0	; select memory bank 0
; set inputs/outputs
	movlw	B'00000000'
	movwf	GPIO			; ports low
	movlw	B'00000111'		; comparators off
	movwf	CMCON
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000000'		; pullups off
	movwf	WPU
	movlw	B'00011111'		; outputs/inputs set 
	movwf	TRISIO			; port data direction register
	movlw	B'01000101'		; settings (pullups disabled) timer0 /64 prescaler (bit 6: 1 is rising edge, 0 is falling edge interrupt) 4V and 1V triggers
	movwf	OPTION_REG
; analog inputs, A/D
	movlw	B'01011011'
	movwf	ANSEL			; digital I/O and analog at AN0

	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'00000000'		; channel 0, left justified, VDD ref etc

;bits 3-2 CHS1:CHS0: Analog Channel Select bits
;00 = Channel 00 (AN0) CURRENT
;01 = Channel 01 (AN1) SPEED POT
;11 = Channel 03 (AN3) FEEDBACK Gain

	movwf	ADCON0
	bsf		ADCON0,0	; A/D on

;Timer1 on
	movlw 	B'00110001'		; divide by 8 for 8us per count
	movwf	T1CON

;  initial states
	clrf		SPEED0		; speed setting by VR1 ms byte
	clrf		SPEED1		; speed setting by VR1 ls byte
	clrf		SOFTSTART		; soft start flag cleared at startup and after speed pot returned to minimum
	movlw	H'FF'
	movwf	SOFT_RUN		; start at end of cycle ramping up to  SPEED1
	movlw	D'1'
	movwf	DEC_RATE		; soft start rate
	clrf		SPEEDPOT		; speed pot read flag
	clrf		TMR1L			; timer1 clear
	clrf		TMR1H
	movlw	D'10'
	movwf	INTER_STRT		; interrupt start counter
	clrf		CYCLE_9		; count waveform cycle 
;	clrf		GATE_OFF		; gate off flag
	movlw	D'5'
	movwf	SOFT_EXTN		; soft start extension

; Startup timer
	movlw	D'100'
	movwf	TEMP
STRT_DEL
	call		DELAYms		; 20ms approx
	decfsz	TEMP,f
	goto	STRT_DEL		; startup delay

;_________________________________________________________________________________
;Check if soft start required (changed with full gain and speed at startup plus feedback current >D'50')
; FEEDBACK Gain
;11 = Channel 03 (AN3) 
	bsf		ADCON0,2
	bsf		ADCON0,3		; channel 3
	call		DELAYX
	call		ACQUIRE_AD
	bcf		STATUS,C
	comf	ADRESH,w
	btfss	STATUS,Z
	goto	RD_EEP		; read EEPROM

; SPEED POT
;01 = Channel 01 (AN1) 
	bsf		ADCON0,2		; channel 1
	bcf		ADCON0,3		;
	call		DELAYX
	call		ACQUIRE_AD
	comf	ADRESH,w		; 8-bit msb inverted
	btfss	STATUS,Z
	goto	RD_EEP		; read EEPROM

;FEEDBACK CURRENT
	bcf		ADCON0,2		; 
	bcf		ADCON0,3		; channel 0
	call		DELAYX
	call		ACQUIRE_AD
	movf	ADRESH,w
	sublw	D'50'			; needs to be more than 1V
	btfsc	STATUS,C
	goto	RD_EEP

; alternate value from 0 to 1 or 1 to 0 
	movlw	EEPROM0
	call		EEREAD
	movwf	SOFT
	movf	SOFT,w
	btfss	STATUS,Z		; if zero set as 1
	goto	CLEAR
	
	movlw	D'1'	
	call		EEWRITE
	goto	RD_EEP
CLEAR
	clrw		
	call		EEWRITE

RD_EEP
	movlw	EEPROM0
	call		EEREAD
	movwf	SOFT
;_________________________________________________________________________________

; Interrupts
	bsf		STATUS,RP0	; select memory bank 1
	bsf		IOC,2			; port change on GP2		
	bcf		STATUS,RP0	; select memory bank 0

	bsf		INTCON,GPIE	; enable port change interrrupt
	bcf		INTCON,GPIF	;  flag	 
	bsf 		INTCON,GIE		; set global interrupt enable 

UPDATE
; read VR1, VR2 and feedback
;bits 3-2 CHS1:CHS0: Analog Channel Select bits

; Read feedback gain trim pot
; FEEDBACK Gain
;11 = Channel 03 (AN3) 
	bsf		ADCON0,2
	bsf		ADCON0,3		; channel 3
	call		DELAYX
	call		ACQUIRE_AD
	bcf		STATUS,C
	rrf		ADRESH,w
	movwf	GAIN

;00 = Channel 00 (AN0) 
;FEEDBACK CURRENT
	bcf		ADCON0,2		; 
	bcf		ADCON0,3		; channel 0
	call		DELAYX
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	FEEDBACK		; feedback from current transformer

; soft start with zero current

	btfss	STATUS,Z		; if clear start softstart
	goto	SPEED_POT_AD
; bypass if this is not required
	movf	SOFT,w
	btfss	STATUS,Z
	goto	SPEED_POT_AD

	clrf		SOFTSTART		; soft start flag cleared at startup and after speed pot returned to minimum
	movlw	D'191'			; 135 degrees max
	movwf	SOFT_RUN		; start at end of cycle ramping up to  SPEED1
	movlw	D'1'
	movwf	DEC_RATE		; soft start rate

SPEED_POT_AD	
; SPEED POT
;01 = Channel 01 (AN1) 
	bsf		ADCON0,2		; channel 1
	bcf		ADCON0,3		;
	call		DELAYX
	call		ACQUIRE_AD
	comf	ADRESH,w		; 8-bit msb inverted
	movwf	SPEED1		; ls byte
	bsf		GATE_OFF,0	; gate off flag set so gate can be driven

; if softstart is clear, reduce startup from FF to SPEED1  

	movf	SOFTSTART,w
	btfss	STATUS,Z
	goto	LS_BITS_RUN
; compare with SPEED1
	movf	SPEED1,w
	subwf	SOFT_RUN,w
	btfss	STATUS,C
	goto	END_SOFT		; use unchanged SPEED1	

; check if SOFT_RUN is zero
	movf	SOFT_RUN,w
	btfsc	STATUS,Z		; if zero then use unchanged SPEED1 and set SOFTSTART flag to end soft start
	goto	END_SOFT

DEC_LOOP
	decfsz	SOFT_EXTN,f
	goto	TRANSFER
	movlw	D'7'
	movwf	SOFT_EXTN		; soft start extension

	decfsz	SOFT_RUN,f
	goto	REDUCE
	goto	END_SOFT		; if zero then use unchanged SPEED1 and set SOFTSTART flag to end soft start
REDUCE	
	decfsz	DEC_RATE,f		; softstart reduce rate
	goto	DEC_LOOP
TRANSFER
; transfer SOFT_RUN to SPEED1
	movf	SOFT_RUN,w
	movwf	SPEED1
; reload soft start rate ** change for soft start rate and soft_extn if necessary
	movlw	D'1'				; 
	movwf	DEC_RATE		; soft start rate
	goto	LS_BITS_RUN
	
END_SOFT; end of soft start
	movlw	D'01'
	movwf	SOFTSTART
LS_BITS_RUN
; get ls bits from A/D
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w		; bits 9 and 10 lsb
	bcf		STATUS,RP0	; select memory bank 0
	movwf	TEMP			; lsb
	comf	TEMP,f			; complement value for reverse direction
	clrf		SPEED0	

; shift TEMP into SPEED and then SPEED0 for 10-bit 1024 value range
	rlf		TEMP,f
	rlf		SPEED1,f
	rlf		SPEED0,f	
	rlf		TEMP,f
	rlf		SPEED1,f
	rlf		SPEED0,f	
;  range in 8us steps

; due to zero voltage phase lag, set maximum value so pulses are stopped 400us plus the 100us gate pulse before next zero crossing detection

	movf	SPEED0,w		; ms byte of speed setting
	movwf	AARGB0	
	movf	SPEED1,w		; ls byte
	movwf	AARGB1

; multiply by initial calibration values in MULT0 and MULT1 then divide by 1024
	movf	MULT0,w
	movwf	BARGB0
	movf	MULT1,w
	movwf	BARGB1

	movwf	BARGB1
	call 		FXM1616U		; multiply
; result in AARGB0,1,2,3
; divide by 1024 by using AARGB1 and AARGB2 for divide by 256 and rotating right by two steps
	bcf		STATUS,C
	rrf		AARGB1,f
	rrf		AARGB2,f
	bcf		STATUS,C
	rrf		AARGB1,w
	movwf	SPEED0		; speed ms byte
	rrf		AARGB2,w		; place in SPEED
	movwf	SPEED1		; ls byte
;....................................................................................................................................................
; do feedback calculations. 

; FEEDBACK is 8-bit value of current
; GAIN is amount of feedback to be applied to speed
	clrf		AARGB0		; ms bytes cleared
	clrf		BARGB0

; clear GAIN when in softstart
	movf	SOFTSTART,w
	btfss	STATUS,Z		; if softstart use 0 for gain
	goto	GAIN_GAIN
	clrw	; use 0 for gain
	goto	GAIN0
GAIN_GAIN; use gain value
	movf	GAIN,w			; load ls bytes

GAIN0
	movwf	AARGB1
	movf	FEEDBACK,w
	movwf	BARGB1
	call 		FXM1616U		; multiply
; result in AARGB,2,3
; divide by  rotating right 
	bcf		STATUS,C
	rrf		AARGB2,f
	rrf		AARGB3,f		; /2
	bcf		STATUS,C
	rrf		AARGB2,f		;  ms byte
	rrf		AARGB3,f		;  ls byte /4
	bcf		STATUS,C
	rrf		AARGB2,f
	rrf		AARGB3,f		; /8
	bcf		STATUS,C
	rrf		AARGB2,f		;  ms byte
	rrf		AARGB3,f		;  ls byte /16

; subtract from SPEED0,1

	movf   	AARGB3,w 	 	 ; low byte 
     	subwf   	SPEED1,f    		 ; subtract low bytes
    
      	movf	AARGB2,w 		; high byte
     	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
    	subwf   	SPEED0,f     		; subtract high bytes
    
     	btfsc   	STATUS,C     	; bypass if SPEED> AARGB
      	goto    	SUB_VAL        	; use subtracted value
; clear SPEED
	clrf		SPEED0
	clrf		SPEED1
SUB_VAL; subtracted value or 0
;......................................................................................................................................................
;  65536 minus speed for gate_on
	movf	SPEED0,w
	sublw	H'FF'			; ms byte subtract
	movwf	TEMP
	movf	SPEED1,w
	sublw	H'FF'
;
; stop interrupt
	bcf		INTCON,GIE
	movwf	GATE_ON1		; ls byte
	btfss	STATUS,C
	decf	TEMP,f

	movf	TEMP,w
	movwf	GATE_ON0		; ms byte
; re-allow interrupt
	bsf		INTCON,GIE

	bsf		SPEEDPOT,0	; speed pot read flag. Set flag as having been read
	goto	UPDATE

; *****************************************************************************************************************************************************
; subroutines

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return

; delay loop 

DELAYms ; approx 20ms
	movlw	D'23'		; delay value
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'117'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8
	return

DELAYX
	movlw	D'30'
	movwf	STORE1		; STORE1 is  loop value	
LOOP1
	decfsz	STORE1,f
	goto	LOOP1
	return

; read mode
EEREAD
	bsf 		STATUS,RP0 	; Bank 1
	movlw 	EEPROM0		;
	movwf 	EEADR 			; Address to read
	bsf 		EECON1,RD 	; EE Read
	movf 	EEDATA,w 		; Move data to w
	bcf		STATUS,RP0	; Bank 0
	return

; EEPROM write
EEWRITE
	bsf 		STATUS,RP0 	; Bank 1
	movwf 	EEDATA		; data
	movlw 	EEPROM0		;
	movwf 	EEADR 			; Address
	bsf 		EECON1,WREN ; Enable write
	movlw 	H'55'			; Unlock write
	movwf 	EECON2 		;
	movlw 	H'AA' 	
	movwf 	EECON2 ;
	bsf 		EECON1,WR 	; Start the write
	bcf		STATUS,RP0	; Bank 0
	bcf		PIR1,EEIF		; clear end of write bit
WRITE
	btfsc	PIR1,EEIF
	goto	READY_RETURN
	goto	WRITE
READY_RETURN
	bsf 		STATUS,RP0 	; Bank 1
	bcf 		EECON1,WREN	 ; disable write
	bcf		STATUS,RP0	; Bank 0
	return


; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;
;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0


FXM1616U 
	        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A    
		 RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B    
		 RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   H'00'
BLUM1616NAP     
		BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP    
		 BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     
		RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA     
		 RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616   
		 RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      
		RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN


	end
